from luma.core.render import canvas
from PIL import ImageFont
import config

# --- 动态加载驱动 ---
if config.USE_EMULATOR:
    from luma.emulator.device import pygame
else:
    from luma.core.interface.serial import i2c
    from luma.oled.device import ssd1306

class OledDisplay:
    def __init__(self):
        try:
            if config.USE_EMULATOR:
                print("🖥️  Mode: PC Emulator (Pygame)")
                self.device = pygame(width=128, height=64)
            else:
                print("📟 Mode: Hardware OLED (I2C)")
                serial = i2c(port=config.I2C_PORT, address=config.I2C_ADDRESS)
                self.device = ssd1306(serial)
            self.font = ImageFont.load_default()
        except Exception as e:
            print(f"❌ Display Init Failed: {e}")
            self.device = None

    # --- 页面 1: 总览 ---
    def draw_page_1(self, draw, data):
        draw.text((0, 0), f"HOST: {data['hostname']}", fill="white")
        draw.line((0, 11, 128, 11), fill="white")
        draw.text((0, 14), f"IP: {data['ip']}", fill="white")
        
        # CPU 条
        draw.text((0, 28), f"CPU: {data['cpu_total']}%", fill="white")
        w = int((data['cpu_total'] / 100) * 60)
        draw.rectangle((65, 28, 65+w, 36), outline="white", fill="white")
        
        # MEM 条
        draw.text((0, 40), f"MEM: {data['mem']}%", fill="white")
        w = int((data['mem'] / 100) * 60)
        draw.rectangle((65, 40, 65+w, 48), outline="white", fill="white")
        
        draw.point((120, 62), fill="white") # Page Indicator

    # --- 页面 2: 详情 ---
    def draw_page_2(self, draw, data):
        draw.text((0, 0), "SYSTEM INFO", fill="white")
        draw.line((0, 11, 128, 11), fill="white")
        draw.text((0, 15), f"TEMP: {data['temp']}", fill="white")
        draw.text((0, 28), f"BOOT: {data['boot_date']}", fill="white")
        draw.text((0, 40), f"UP: {data['uptime']}", fill="white")
        draw.point((123, 62), fill="white")

    # --- 页面 3: 核心矩阵 ---
    def draw_page_3(self, draw, data):
        draw.text((0, 0), "CORE MATRIX", fill="white")
        draw.line((0, 11, 128, 11), fill="white")
        
        cores = data['cpu_cores']
        cols = 8; box_w = 12; box_h = 8; gap = 3
        start_x = 2; start_y = 16
        
        for i, usage in enumerate(cores):
            row = i // cols
            col = i % cols
            x = start_x + col * (box_w + gap)
            y = start_y + row * (box_h + gap)
            if y + box_h > 60: break
            
            if usage > 50:
                draw.rectangle((x, y, x+box_w, y+box_h), fill="white")
            else:
                draw.rectangle((x, y, x+box_w, y+box_h), outline="white")
                
        draw.point((126, 62), fill="white")

    # --- 页面 4: 网络与磁盘 ---
    def draw_page_4(self, draw, data):
        draw.text((0, 0), "NET & DISK", fill="white")
        draw.line((0, 11, 128, 11), fill="white")
        
        net = data['net']
        fmt = lambda k: f"{k/1024:.1f}M" if k>1024 else f"{int(k)}K"
        
        draw.text((0, 16), f"D:{fmt(net['down'])}/s", fill="white")
        draw.text((64, 16), f"U:{fmt(net['up'])}/s", fill="white")
        
        draw.text((0, 32), f"DISK: {data['disk']}%", fill="white")
        w = int((data['disk'] / 100) * 120)
        draw.rectangle((0, 44, 120, 50), outline="white")
        draw.rectangle((0, 44, w, 50), fill="white")
        draw.point((128, 62), fill="white")

    def render(self, data, page_index):
        with canvas(self.device) as draw:
            if data.get('alert'):
                draw.rectangle((0,0,128,64), fill="white")
                draw.text((25, 25), "!! ALERT !!", fill="black")
                return
            
            pages = [self.draw_page_1, self.draw_page_2, self.draw_page_3, self.draw_page_4]
            if page_index < len(pages):
                pages[page_index](draw, data)

    def draw_offline(self):
        with canvas(self.device) as draw:
            draw.text((30, 20), "OFFLINE", fill="white")
            draw.text((10, 35), "Check Network", fill="white")
